using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using System.Reflection;
using gov.va.med.vbecs.Common;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/12/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Maximum Surgical Blood Orderinng Schedule (MSBOS) business layer class
		///</summary>

		#endregion


	public class Msbos : BaseBusinessObject, IRecordStatusCode
	{
		/// <summary>
		/// Type of Procedure
		/// </summary>
		private string _surgeryName;
		/// <summary>
		/// MSBOS unique identifier
		/// </summary>
		private System.Guid _msbosGuid;
		/// <summary>
		/// Component classes defined for MSBOS procedure
		/// </summary>
		private DataRow[] _bloodComponentClasses;
		/// <summary>
		/// Indicates is procedure is type and screen and requires no component classes
		/// </summary>
		private bool _isTypeAndScreen;
//		/// <summary>
//		/// Indicates if the MSBOS procedure has component classes
//		/// </summary>
//		//private bool _hasComponentClasses;
		/// <summary>
		/// Indicates if procedure is not type and screen but requires no component classes
		/// </summary>
		private bool _isNoBloodRequired;
		/// <summary>
		/// Code to indicate if MSBOS procedure is active
		/// </summary>
		private Common.RecordStatusCode _recordStatusCode;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="888"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Initialized MSBOS object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="360"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSBOS class constructor
		/// </summary>
		public Msbos()
		{
			this._surgeryName = String.Empty;
			this._msbosGuid = System.Guid.NewGuid();
			this._isTypeAndScreen = false;
			this._isNoBloodRequired = false;
			this._recordStatusCode = Common.RecordStatusCode.Active;
   
			this.IsNew = true;
			this.IsDirty = true;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2411"> 
		///		<ExpectedInput>DataRow containing MSBOS data</ExpectedInput>
		///		<ExpectedOutput>MSBOS object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="363"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates MSBOS object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public Msbos(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="889"> 
		///		<ExpectedInput>Initialized MSBOS object</ExpectedInput>
		///		<ExpectedOutput>Surgery name of MSBOS object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="890"> 
		///		<ExpectedInput>Valid surgery name string</ExpectedInput>
		///		<ExpectedOutput>Updated MSBOS object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="891"> 
		///		<ExpectedInput>Invalid surgery name string</ExpectedInput>
		///		<ExpectedOutput>Format Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets name of surgical procedure
		/// </summary>
		public string SurgeryName
		{
			get
			{
				return _surgeryName;
			}
			set
			{
				if (value.Trim().Length > 50)
				{
					base.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), "50").ResString;
					throw new BusinessObjectException(errorMessage);
				}

				this._surgeryName = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, ((value.Trim().Length == 0)||(value.Trim().Length > 50)));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="892"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="893"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets MSBOS unique identifier
		/// </summary>
		public System.Guid MsbosGuid
		{
			get
			{
				return _msbosGuid;
			}
			set
			{
				this._msbosGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="894"> 
		///		<ExpectedInput>Initialized MSBOS object</ExpectedInput>
		///		<ExpectedOutput>Collection of data rows containing blood component class info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="895"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets data row array or blood component classes associated with MSBOS surgical procedure
		/// </summary>
		public DataRow[] BloodComponentClasses
		{
			get
			{
				return _bloodComponentClasses;
			}
			set
			{
				if ((_bloodComponentClasses == null) || (value != _bloodComponentClasses))
				{
					this._bloodComponentClasses = value;
					this.IsDirty = true;
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="896"> 
		///		<ExpectedInput>Initialized MSBOS object</ExpectedInput>
		///		<ExpectedOutput>Type and screen indicator for MSBOS object</ExpectedOutput>
		///	</Case>
		///
		///	<Case type="1" testid ="897"> 
		///		<ExpectedInput>true with componentclasses</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// Gets or sets value indicating whether a surgical procedure is a type and screen.
		/// Type and screen procedures have no associated blood component classes
		/// </summary>
		public bool IsTypeAndScreen
		{
			get
			{
				return _isTypeAndScreen;
			}
			set
			{
				bool _broken = false;

				if (this._bloodComponentClasses != null)
				{
					if (this._bloodComponentClasses.Length > 0)
					{
						if (this._isTypeAndScreen)
						{
							RuleBroken(MethodBase.GetCurrentMethod().Name,true);
							_broken = true;
						}
					}
				}

				if (!_broken)
				{
					this._isTypeAndScreen = value;
					RuleBroken(MethodBase.GetCurrentMethod().Name,false);
					this.IsDirty = true;
				}
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6334"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6335"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether a surgical procedure requires no blood components.
		/// </summary>
		public bool IsNoBloodRequired
		{
			get
			{
				return _isNoBloodRequired;
			}
			set
			{
				bool _broken = false;

				if (this._bloodComponentClasses != null)
				{
					if (this._bloodComponentClasses.Length > 0)
					{
						if (this._isNoBloodRequired)
						{
							RuleBroken(MethodBase.GetCurrentMethod().Name,true);
							_broken = true;
						}
					}
				}

				if (!_broken)
				{
					this._isNoBloodRequired = value;
					RuleBroken(MethodBase.GetCurrentMethod().Name,false);
					this.IsDirty = true;
				}
			}
		}
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="898"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="899"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating active status of MSBOS surgical procedure
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return this._recordStatusCode;
			}
			set
			{
				this._recordStatusCode = value;
				this.IsDirty = true;
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		/// <summary>
		/// Method to load MSBOS class properties from a given data row
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			this.SurgeryName = dtRow[TABLE.Msbos.SurgeryName].ToString();
			
			this.MsbosGuid = dtRow.IsNull(TABLE.Msbos.MsbosGuid) ? Guid.Empty : new Guid(dtRow[TABLE.Msbos.MsbosGuid].ToString());
			this.IsTypeAndScreen = dtRow.IsNull(TABLE.Msbos.TypeAndScreenIndicator) ? false : (bool)dtRow[TABLE.Msbos.TypeAndScreenIndicator];
			this.IsNoBloodRequired = dtRow.IsNull(TABLE.Msbos.NoBloodRequiredIndicator) ? false : (bool)dtRow[TABLE.Msbos.NoBloodRequiredIndicator];
			this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.Msbos.RecordStatusCode].ToString());
			
			if (!dtRow.IsNull(TABLE.Msbos.RowVersion))
			{
					this.RowVersion = (byte[])dtRow[TABLE.Msbos.RowVersion];
			}

			this.IsDirty = false;
			this.IsNew = false;
		}
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1222"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="375"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="901"> 
		///		<ExpectedInput>MSBOS object</ExpectedInput>
		///		<ExpectedOutput>Updated MSBOS object with properties reset</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="364"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public method to reset properties of MSBOS class to initial values
		/// </summary>
		public void Clear()
		{
			this._surgeryName = string.Empty;
			this._msbosGuid = Guid.Empty;
			this._isTypeAndScreen = false;
			this._isNoBloodRequired = false;
			this._recordStatusCode = Common.RecordStatusCode.Unknown;
			this.IsDirty = false;
			this.IsNew = true;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="902"> 
		///		<ExpectedInput>Data set of valid MSBOS and MSBOS component class info</ExpectedInput>
		///		<ExpectedOutput>Success indicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2001"> 
		///		<ExpectedInput>Invalid MSBOS and MSBOS component class data set</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to save a dataset containing MSBOS entries and return success status
		/// </summary>
		/// <param name="msbosComponents">Data set containing msbos information</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Success indicator</returns>
		public bool Save(DataSet msbosComponents, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList _dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList _sprocArrayList = new System.Collections.ArrayList();

			DataTable dtMsbosUpdates = msbosComponents.Tables[0].GetChanges(DataRowState.Modified);
			if (dtMsbosUpdates != null)
			{
				_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtMsbosUpdates, lastUpdateFunctionId));
				_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateMsbos.StoredProcName);
			}

			DataTable dtMsbosInserts = msbosComponents.Tables[0].GetChanges(DataRowState.Added);
			if (dtMsbosInserts != null)
			{
				_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtMsbosInserts, lastUpdateFunctionId));
				_sprocArrayList.Add(Common.VbecsStoredProcs.InsertMsbos.StoredProcName);
			}

			DataTable dtCCUpdates = msbosComponents.Tables[1].GetChanges(DataRowState.Modified);
			if (dtCCUpdates != null)
			{
				if (dtCCUpdates.Columns.Contains(TABLE.ComponentClass.ComponentClassName))
					dtCCUpdates.Columns.Remove(TABLE.ComponentClass.ComponentClassName);

				if (dtCCUpdates.Columns.Contains(TABLE.MsbosComponentClass.ComponentClassId))
					dtCCUpdates.Columns.Remove(TABLE.MsbosComponentClass.ComponentClassId);

				_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtCCUpdates, lastUpdateFunctionId));
				_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateMsbosComponentClass.StoredProcName);
			}

			System.Data.DataTable dtCCInserts = msbosComponents.Tables[1].GetChanges(DataRowState.Added);
			if (dtCCInserts != null)
			{
				if (dtCCInserts.Columns.Contains(TABLE.ComponentClass.ComponentClassName))
					dtCCInserts.Columns.Remove(TABLE.ComponentClass.ComponentClassName);

				if (dtCCInserts.Columns.Contains(TABLE.Msbos.RowVersion))
					dtCCInserts.Columns.Remove(TABLE.Msbos.RowVersion);

				_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtCCInserts, lastUpdateFunctionId));
				_sprocArrayList.Add(Common.VbecsStoredProcs.InsertMsbosComponentClass.StoredProcName);
			}

			return DAL.Msbos.Save(_sprocArrayList, _dtArrayList);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="903"> 
		///		<ExpectedInput>Boolean value</ExpectedInput>
		///		<ExpectedOutput>Data table of orderable components for division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3112"> 
		///		<ExpectedInput>MSBOS object with no division code set</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to get all component classes defined as orderable for division (BR_32.02)
		/// </summary>
		/// <returns>Data table containing component classes</returns>
		/// 
		public DataTable GetOrderableComponentClasses()
		{
			DataTable dt = DAL.Msbos.GetOrderableComponentClasses();
			DataTable returnTable = dt.Clone();
			
			foreach(DataRow dr in dt.Rows)
			{
				//BR_32.02
				if (((int)dr[TABLE.ComponentClass.ComponentClassId] == (int)Common.ComponentClass.RBC) ||
					((int)dr[TABLE.ComponentClass.ComponentClassId] == (int)Common.ComponentClass.WB) ||
					((int)dr[TABLE.ComponentClass.ComponentClassId] == (int)Common.ComponentClass.FFP) ||
					((int)dr[TABLE.ComponentClass.ComponentClassId] == (int)Common.ComponentClass.PLT) ||
					((int)dr[TABLE.ComponentClass.ComponentClassId] == (int)Common.ComponentClass.CRYO))
				{
					returnTable.Rows.Add(dr.ItemArray);
				}
			}
			return returnTable;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="904"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data set containing MSBOS and MSBOS component classes info</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="374"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function which calls data access layer to retrieve MSBOS data for division
		/// </summary>
		/// <returns>Data set containing MSBOS and MSBOS Component Class information</returns>
		public DataSet GetMsbosEntries()
		{
			return DAL.Msbos.GetMsbosEntries();
		}
	}
}
